/*
 * Copyright (c) 2005, 2006, Regents of the University of California
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.  
 *
 * * Neither the name of the University of California, Berkeley nor
 *   the names of its contributors may be used to endorse or promote
 *   products derived from this software without specific prior 
 *   written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package blog;

import java.util.*;

/**
 * A number random variable.  It consists of a potential object pattern (POP) 
 * and a tuple of generating objects.  Its value in a given world is the 
 * number of objects generated by that POP and those generating objects.
 */
public class NumberVar extends VarWithDistrib {
    /**
     * Creates a new NumberVar for the given POP and given tuple of 
     * generating objects.
     */
    public NumberVar(POP pop, List args) {
	super(args);
	this.pop = pop;
    }

    /**
     * Creates a new NumberVar for the given POP and given tuple of 
     * generating objects.
     */
    public NumberVar(POP pop, Object[] args) {
	super(args);
	this.pop = pop;
    }

    /**
     * Creates a new NumberVar for the given POP and given tuple of 
     * generating objects.  If <code>stable</code> is true, then the 
     * caller guarantees that the given <code>args</code> array will not 
     * be modified externally. 
     */
    public NumberVar(POP pop, Object[] args, boolean stable) {
	super(args, stable);
	this.pop = pop;
    }

    /**
     * Returns the potential object pattern associated with this number 
     * variable.
     */
    public final POP pop() {
	return pop;
    }
    
    public DependencyModel.Distrib getDistrib(EvalContext context) {
	context.pushEvaluee(this);
	DependencyModel.Distrib distrib 
	    = pop.getDepModel().getDistribWithBinding
	    (context, pop.getGenObjVars(), args(), ZERO);
	context.popEvaluee();
	return distrib;
    }

    /**
     * Returns the integer type.
     */
    public Type getType() {
	return BuiltInTypes.NATURAL_NUM;
    }

    public int getOrderingIndex() {
	return pop.getDepModel().getCreationIndex();
    }

    public DependencyModel getDepModel() {
	return pop.getDepModel();
    }

    /**
     * Returns the value that the given origin function yields on all 
     * objects that satisfy this number variable.  This is either one of 
     * this variable's arguments or Model.NULL.
     *
     * @throws IllegalArgumentException if the argument type of the given 
     *                                  origin function is not the type 
     *                                  of objects that satisfy this variable
     */
    public Object getOriginFuncValue(OriginFunction g) {
	if (g.getArgType() != pop.type()) {
	    throw new IllegalArgumentException
		("Can't evaluate origin function " + g + " on objects "
		 + "of type " + pop.type());
	}

	int index = pop.getOriginFuncIndex(g);
	if (index == -1) {
	    return Model.NULL;
	}
	return args[index];
    }

    /**
     * Returns the number of objects generated by this POP on this tuple 
     * of generating objects in the given world.
     */
    public int numGenerated(PartialWorld w) {
	return ((Integer) getValue(w)).intValue();
    }

    public Object clone() {
	return new NumberVar(pop, args);
    }

    /**
     * Two NumberVar objects are equal if they have the same POP and
     * their argument lists are equal (recall that list equality is
     * checked by calling the <code>equals</code> method on each
     * corresponding pair of objects in the two lists).
     */
    public boolean equals(Object obj) {
	if (obj instanceof NumberVar) {
	    NumberVar other = (NumberVar) obj;
	    return ((pop == other.pop()) 
                    && Arrays.equals(args, other.args()));
	}
	return false;
    }
		
    public int hashCode() {
        int code = pop.hashCode();
        for (int i = 0; i < args.length; ++i) {
            code ^= args[i].hashCode();
        }
	return code;
    }

    /**
     * Returns a string of the form #Type(f1 = o1, ..., fK = oK) 
     * where f1, ..., fK are the origin functions and o1, ..., oK are 
     * the generating objects.
     */
    public String toString() {
	StringBuffer buf = new StringBuffer();
	buf.append("#");
	buf.append(pop.type());
		
	OriginFunction[] originFuncs = pop.originFuncs();
	if (originFuncs.length > 0) {
	    buf.append("(");
	    for (int i = 0; i < originFuncs.length; ++i) {
		buf.append(originFuncs[i]);
		buf.append(" = ");
		buf.append(args[i]);
		if (i + 1 < originFuncs.length) {
		    buf.append(", ");
		}
	    }
	    buf.append(")");
	}

	return buf.toString();
    }	

    private POP pop;

    private static final Integer ZERO = new Integer(0);
}
