/*
 * Copyright (c) 2006, Regents of the University of California
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.  
 *
 * * Neither the name of the University of California, Berkeley nor
 *   the names of its contributors may be used to endorse or promote
 *   products derived from this software without specific prior 
 *   written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package blog;

import java.util.*;

/**
 * Implementation of the EvalContext interface that records the basic 
 * random variables used by calls to the access methods.  The 
 * <code>getParents</code> method returns this set of variables.
 */
public class ParentRecEvalContext extends DefaultEvalContext {
    /**
     * Creates a new ParentRecEvalContext using the given world.
     */
    public ParentRecEvalContext(PartialWorld world) {
	super(world);
    }

    /**
     * Creates a new ParentRecEvalContext using the given world.  
     * If the <code>errorIfUndet</code> flag is true, the access
     * methods on this instance will print error messages and exit the
     * program if the world is not complete enough to determine the
     * correct return value.  Otherwise they will just return null in 
     * such cases.  
     */
    public ParentRecEvalContext(PartialWorld world, boolean errorIfUndet) {
	super(world, errorIfUndet);
    }

    public Object getValue(BasicVar var) {
	Object value = world.getValue(var);
	if (value == null) {
	    latestUninstParent = var;
	    var.ensureStable();
	    handleMissingVar(var);
	} else {
	    if (parents.add(var)) {
		var.ensureStable();
	    }
	}
	return value;
    }

    // Note that we don't have to override getSatisfiers, because the 
    // DefaultEvalContext implementation of getSatisfiers calls getValue 
    // on the number variable

    public NumberVar getPOPAppSatisfied(Object obj) {
	if (obj instanceof NonGuaranteedObject) {
	    return ((NonGuaranteedObject) obj).getNumberVar();
	} 

	if (obj instanceof ObjectIdentifier) {
	    parents.add(new OriginVar((ObjectIdentifier) obj));
	    return world.getPOPAppSatisfied(obj);
	}

	// Must be guaranteed object, so not generated by any number var
	return null;
    }

    /**
     * Returns the set of basic random variables that are instantiated
     * and whose values have been used in calls to the access methods.
     * This set is backed by the ParentRecEvalContext and will change
     * as more random variables are used.
     *
     * @return unmodifiable Set of BasicVar 
     */
    public Set getParents() {
	return Collections.unmodifiableSet(parents);
    }

    /**
     * Returns the variable whose value was most recently needed by an access 
     * method, but which is not instantiated.  This method returns null 
     * if no such variable exists.  
     */
    public BasicVar getLatestUninstParent() {
	return latestUninstParent;
    }

    protected Set parents = new LinkedHashSet(); // of BasicVar
    protected BasicVar latestUninstParent = null;
}
	
