/*
 * Copyright (c) 2005, Regents of the University of California
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.  
 *
 * * Neither the name of the University of California, Berkeley nor
 *   the names of its contributors may be used to endorse or promote
 *   products derived from this software without specific prior 
 *   written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package blog;

import java.util.*;

import common.Util;

/**
 * A Particle for Particle Filtering.
 * The particle is advanced as it receives evidence and queries with calls to methods {@link #take(Evidence)} and {@link #answerQueries(Collection)}.
 * Providing evidence also reweights the particle according to it.
 * A method {@link #copy()} can be used for resampling.
 */
public class Particle {

    /**
     * Creates a new particle.
     * <code>numTimeSlicesInMemory</code> indicates how many time slices need to be kept in memory.
     * The properties table specifies configuration parameters.  
     */
    public Particle(Set idTypes, int numTimeSlicesInMemory) {
	this.numTimeSlicesInMemory = numTimeSlicesInMemory;
	this.idTypes = idTypes;
	curWorld = new DefaultPartialWorld(idTypes);
	weight = -1;
    }

    /**
     * Takes evidence relative to some new time step,
     * updates current world based on it,
     * and recalculates particle weight according to its probability.
     */
    public void take(Evidence evidence) {
	setValues(evidence);
	ensureVariablesAreSupported(evidence.getEvidenceVars());
	weight = evidence.getEvidenceProb(curWorld);
    }

    /**
     * Takes a collection of queries and answers them based on current world.
     */
    public void answerQueries(Collection queries) {
	Collection queriesVars = getQueriesVars(queries);
	includeDerivedAmongVars(queriesVars);
	ensureVariablesAreSupported(queriesVars);
	updateQueriesStats(queries);
    }

    /**
     * Sets given evidence in current world.
     */
    private void setValues(Evidence evidence) {
	for (Iterator iter = evidence.getEvidenceVars().iterator(); iter.hasNext(); ) {
	    BayesNetVar var = (BayesNetVar) iter.next();
	    if (var instanceof DerivedVar)
		curWorld.addDerivedVar((DerivedVar) var);
	    else
		curWorld.setValue((BasicVar) var, evidence.getObservedValue(var));
	}
    }

    public PartialWorld getLatestWorld() {
	if (curWorld == null) {
	    throw new IllegalStateException
	    ("Particle has no latest sample.");
	}
	return curWorld;
    }

    public double getLatestWeight() {
	if (weight == -1) {
	    throw new IllegalStateException
		("Particle has no latest sample.");
	}
	return weight;
    }

    /**
     * Includes in current world the variables which are derived variables
     * in given collection.
     */
    private void includeDerivedAmongVars(Collection vars) {
	for (Iterator iter = vars.iterator(); iter.hasNext(); ) {
	    BayesNetVar var = (BayesNetVar) iter.next();
	    if (var instanceof DerivedVar)
		curWorld.addDerivedVar((DerivedVar) var);
	}
    }

    /**
     * Ensure given variables are supported in current world.
     */
    private void ensureVariablesAreSupported(Collection vars) {
	InstantiatingEvalContext context 
	    = new InstantiatingEvalContext(curWorld);
	for (Iterator iter = vars.iterator(); iter.hasNext(); ) {
	    BayesNetVar var = (BayesNetVar) iter.next();
	    var.ensureDetAndSupported(context);
	}
    }

    private Collection getQueriesVars(Collection queries) {
	Collection result = new LinkedList();
	for (Iterator iter = queries.iterator(); iter.hasNext(); ) {
	    result.addAll(((Query) iter.next()).getVariables());
	}
	return result;
    }

    public Particle copy() {
	Particle copy = new Particle(idTypes, numTimeSlicesInMemory);
	DefaultPartialWorld newWorld = (DefaultPartialWorld) ((DefaultPartialWorld)curWorld).clone();
	copy.setWorld(newWorld);
	copy.weight = weight;
	return copy;
    }

    protected void setWorld(PartialWorld curWorld) {
	this.curWorld = curWorld;
    }

    public void updateQueriesStats(Collection queries) {
	if(getLatestWeight() > 0) {
	    for(Iterator iter = queries.iterator(); iter.hasNext(); ) {
		Query q = (Query) iter.next();
		q.updateStats(getLatestWorld(), getLatestWeight());
	    }
	}
    }

    private Set idTypes; // of Type
    private PartialWorld curWorld = null;
    private double weight = -1;
    public int numTimeSlicesInMemory;
}
