/*
 * Copyright (c) 2005, Regents of the University of California
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.  
 *
 * * Neither the name of the University of California, Berkeley nor
 *   the names of its contributors may be used to endorse or promote
 *   products derived from this software without specific prior 
 *   written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package blog;

import java.util.*;

import common.Util;

/**
 * A Particle Filter.  
 * It works by keeping a set of {@link Particle}s,
 * each representing a partial world, weighted by the evidence.  The
 * number of particles used is specified by the property
 * <code>numParticles</code> given at construction time (default is
 * <code>100</code>).  If property <code>useDecayedMCMC</code> is set
 * to <code>true</code> (default is <code>false</code>), it uses the
 * rejuvenation method presented by W. R. Gilks and
 * C. Berzuini. "Following a moving target --- Monte Carlo inference
 * for dynamic Bayesian models."  Journal of the Royal Statistical
 * Society, Series B, 63:127--146, 2001.  The number of moves used is
 * specified by the property <code>numMoves</code> given at
 * construction time (default is <code>1</code>).  
 * 
 * <p>The ParticleFilter is an unusual {@link InferenceEngine} in that
 * it takes evidence and queries additional to the ones taken by
 * {@link #setEvidence(Evidence)} and {@link #setQueries(List)}.  The
 * evidence set by {@link #setEvidence(Evidence)} is used in the very
 * beginning of inference (thus keeping the general InferenceEngine
 * semantics for it) and the queries set by {@link #setQueries(List)}
 * are used by {@link #answerQueries()} only (again keeping the
 * original InferenceEngine semantics).
 */
public class ParticleFilter extends InferenceEngine {

    /**
     * Creates a new particle filter for the given BLOG model, with 
     * configuration parameters specified by the given properties table.  
     */
    public ParticleFilter(Model model, Properties properties) {
	super(model);

	String numParticlesStr = properties.getProperty("numParticles", "1000");
	try {
	    numParticles = Integer.parseInt(numParticlesStr);
	} catch (NumberFormatException e) {
	    Util.fatalErrorWithoutStack("Invalid number of samples: " + numParticlesStr); // do not dump stack.
	}

	String useDecayedMCMCStr = properties.getProperty("useDecayedMCMC", "false");
	useDecayedMCMC = Boolean.parseBoolean(useDecayedMCMCStr);
	//if (useDecayedMCMC) numParticles = 1;
	
	String numMovesStr = properties.getProperty("numMoves", "1");
	try {
	    numMoves = Integer.parseInt(numMovesStr);
	} catch (NumberFormatException e) {
	    Util.fatalErrorWithoutStack("Invalid number of moves: " + numMovesStr);
	}

	String idTypesString = properties.getProperty("idTypes", "none");
	idTypes = model.getListedTypes(idTypesString);
	if (idTypes == null) {
	    Util.fatalErrorWithoutStack("Fatal error: invalid idTypes list.");
	}

	if (useDecayedMCMC)
	    dmhSampler = new DMHSampler(model, properties);
    }

    /** Answers the queries provided at construction time. */
    public void answerQueries() {
	resetParticleFilters();
	answer(queries);
    }

    /**
     * Prepares particle filter for a new sequence of evidence and queries by
     * generating a new set of particles from scratch,
     * which are consistent with evidence set by {@link #setEvidence(Evidence)}
     * (if it has not been invoked, empty evidence is assumed),
     * ensuring behavior consistent with other {@link InferenceEngine}s.
     */
    public void resetParticleFilters() {
    	System.out.println("Using " + numParticles + " particles...");
	int numTimeSlicesInMemory = useDecayedMCMC? dmhSampler.getMaxRecall() : 1;
	if(evidence == null)
	    evidence = new Evidence();
	if(queries == null)
	    queries = new LinkedList();
	if (useDecayedMCMC)
	    dmhSampler.initialize(evidence, queries);
	particles = new ArrayList();
	for(int i = 0; i < numParticles; i++) {
	    Particle newParticle = new Particle(idTypes, numTimeSlicesInMemory);
	    newParticle.take(evidence);
	    particles.add(newParticle);
	}
    }

    /** Takes more evidence. {@link #resetParticleFilters()} must have been called. */
    public void take(Evidence evidence) {
	if (particles == null)
	    resetParticleFilters();
	move();
	resample();
	for(Iterator it = particles.iterator(); it.hasNext(); ) {
	    Particle p = (Particle) it.next();
	    p.take(evidence);
	}
    }

    public void answer(Collection queries) {
	for(Iterator it = particles.iterator(); it.hasNext(); ) {
	    Particle p = (Particle) it.next();
	    p.answerQueries(queries);
	}
    }

    private void resample() {
	double[] weights = new double[numParticles];
	boolean[] alreadySampled = new boolean[numParticles];
	double sum = 0.0;
	List newParticles = new ArrayList();

	for(int i=0;i<numParticles;i++) {
	    weights[i] = ((Particle) particles.get(i)).getLatestWeight();
	    sum += weights[i];
	}

	if(Util.withinTol(sum, 0.0)) {
	    throw new IllegalArgumentException
	    ("All particles have zero weight");
	}

	for(int i = 0; i < numParticles; i++) {
	    int selection = Util.sampleWithWeights(weights, sum);
	    if ( ! alreadySampled[selection]) {
		newParticles.add(particles.get(selection));
		alreadySampled[selection] = true;
	    }
	    else newParticles.add(((Particle)particles.get(selection)).copy());
	}

	particles = newParticles;
    }

    private void move() {
	if ( ! useDecayedMCMC) return;

	for(int i = 0; i < numMoves; i++) {
	    for(Iterator iter = particles.iterator(); iter.hasNext(); ) {
		Particle p = (Particle)iter.next();
		p.setWorld(dmhSampler.nextSample(p.getLatestWorld()));
	    }
	}
    }

    private Set idTypes; // of Type

    private int numParticles;
    private boolean useDecayedMCMC;
    private List particles;  // of Particles
    private int numMoves;
    private DMHSampler dmhSampler;
}
